<?php
/**
 * Plugin Name:       DP Country Gate
 * Plugin URI:        https://decisiveplugins.com/plugins/dp-country-gate/
 * Description:       Block or allow access to your WordPress site by visitor country using Cloudflare, server GEOIP headers, or a remote IP API with caching.
 * Version:           1.6.1
 * Author:            Decisive Plugins
 * Author URI:        https://decisiveplugins.com
 * Text Domain:       dp-country-gate
 * Domain Path:       /languages
 * Requires at least: 5.2
 * Tested up to:      6.7
 * Requires PHP:      7.4
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see https://www.gnu.org/licenses/gpl-2.0.html.

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'DDS_Country_Gate' ) ) {

	class DDS_Country_Gate {

		const OPT_KEY          = 'dds_country_gate_options';
		const TRANSIENT_PREFIX = 'dds_country_gate_';

		/**
		 * Constructor.
		 */
		public function __construct() {
			add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );

			add_action( 'admin_init', array( $this, 'register_settings' ) );
			add_action( 'admin_menu', array( $this, 'add_settings_page' ) );

			// Run before most things on the front end.
			add_action( 'template_redirect', array( $this, 'maybe_block_visitor' ), 0 );
		}

		/**
		 * Load plugin textdomain.
		 */
		public function load_textdomain() {
			load_plugin_textdomain(
				'dp-country-gate',
				false,
				dirname( plugin_basename( __FILE__ ) ) . '/languages'
			);
		}

		/**
		 * Check if a Pro add-on is active.
		 *
		 * Pro (or other add-ons) can define DP_COUNTRY_GATE_PRO to true.
		 *
		 * @return bool
		 */
		protected function is_pro_active() {
			return defined( 'DP_COUNTRY_GATE_PRO' ) && DP_COUNTRY_GATE_PRO;
		}

		/**
		 * Default options.
		 *
		 * @return array
		 */
		public function default_options() {
			return array(
				'mode'            => 'blocklist', // blocklist|allowlist.
				'countries'       => '',
				'allow_logged_in' => 1,
				'exempt_roles'    => array(),
				'bypass_paths'    => "/wp-login.php\n/wp-admin/*\n/wp-json/*",
				'use_remote_api'  => 0,
				'cache_ttl'       => 3600, // seconds.
				'block_status'    => 403,  // 403|451|302.
				'block_redirect'  => '',
				'block_message'   => __(
					'<h1>Access Restricted</h1><p>Your country is not permitted to access this site.</p>',
					'dp-country-gate'
				),
			);
		}

		/**
		 * Get merged options.
		 *
		 * @return array
		 */
		public function get_options() {
			$defaults = $this->default_options();
			$saved    = get_option( self::OPT_KEY, array() );

			if ( ! is_array( $saved ) ) {
				$saved = array();
			}

			$options = wp_parse_args( $saved, $defaults );

			/**
			 * Filter the resolved DP Country Gate options array.
			 *
			 * @since 1.6.1
			 *
			 * @param array $options Resolved options.
			 */
			return apply_filters( 'dp_country_gate_options', $options );
		}

		/**
		 * Register settings and fields.
		 */
		public function register_settings() {
			register_setting(
				'dds_country_gate',
				self::OPT_KEY,
				array( $this, 'sanitize_options' )
			);

			add_settings_section(
				'dds_country_gate_main',
				__( 'Country Gate Settings', 'dp-country-gate' ),
				array( $this, 'render_main_section_description' ),
				'dds_country_gate'
			);

			add_settings_field(
				'mode',
				__( 'Mode', 'dp-country-gate' ),
				array( $this, 'field_mode' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'countries',
				__( 'Country Codes', 'dp-country-gate' ),
				array( $this, 'field_countries' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'allow_logged_in',
				__( 'Allow Logged-in Users', 'dp-country-gate' ),
				array( $this, 'field_allow_logged_in' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'exempt_roles',
				__( 'Exempt Roles', 'dp-country-gate' ),
				array( $this, 'field_exempt_roles' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'bypass_paths',
				__( 'Bypass Paths', 'dp-country-gate' ),
				array( $this, 'field_bypass_paths' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'use_remote_api',
				__( 'Remote IP Lookup', 'dp-country-gate' ),
				array( $this, 'field_use_remote_api' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'cache_ttl',
				__( 'Remote Cache Lifetime', 'dp-country-gate' ),
				array( $this, 'field_cache_ttl' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'block_status',
				__( 'Block Response Type', 'dp-country-gate' ),
				array( $this, 'field_block_status' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'block_redirect',
				__( 'Block Redirect URL', 'dp-country-gate' ),
				array( $this, 'field_block_redirect' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);

			add_settings_field(
				'block_message',
				__( 'Block Message', 'dp-country-gate' ),
				array( $this, 'field_block_message' ),
				'dds_country_gate',
				'dds_country_gate_main'
			);
		}

		/**
		 * Section description.
		 */
		public function render_main_section_description() {
			echo '<p>' . esc_html__(
				'Choose whether to block specific countries (blocklist mode) or only allow specific countries (allowlist mode). You can also bypass certain paths, exempt logged-in users or roles, and optionally use a remote IP lookup API with caching.',
				'dp-country-gate'
			) . '</p>';

			echo '<p><em>' . esc_html__(
				'Note: If the remote IP lookup option is enabled, visitor IP addresses will be sent to ipapi.co for country detection. Only the country code is stored in cache.',
				'dp-country-gate'
			) . '</em></p>';
		}

		/**
		 * Sanitize options.
		 *
		 * @param array $input Incoming options.
		 *
		 * @return array
		 */
		public function sanitize_options( $input ) {
			$defaults = $this->default_options();
			$san      = array();

			if ( ! is_array( $input ) ) {
				$input = array();
			}

			// Mode.
			$mode = isset( $input['mode'] ) ? sanitize_text_field( $input['mode'] ) : $defaults['mode'];
			if ( ! in_array( $mode, array( 'blocklist', 'allowlist' ), true ) ) {
				$mode = $defaults['mode'];
			}
			$san['mode'] = $mode;

			// Countries (list of ISO codes, uppercase, comma separated).
			$countries_raw = isset( $input['countries'] ) ? wp_unslash( $input['countries'] ) : '';
			$codes         = array();
			if ( '' !== $countries_raw ) {
				$parts = explode( ',', $countries_raw );
				foreach ( $parts as $part ) {
					$code = strtoupper( trim( $part ) );
					if ( 2 === strlen( $code ) && ctype_alpha( $code ) ) {
						$codes[] = $code;
					}
				}
			}
			$san['countries'] = implode( ',', array_unique( $codes ) );

			// Allow logged-in users.
			$san['allow_logged_in'] = ! empty( $input['allow_logged_in'] ) ? 1 : 0;

			// Exempt roles.
			$san['exempt_roles'] = array();
			if ( ! empty( $input['exempt_roles'] ) && is_array( $input['exempt_roles'] ) ) {
				$editable_roles = function_exists( 'get_editable_roles' ) ? get_editable_roles() : array();
				foreach ( $input['exempt_roles'] as $role_key ) {
					$role_key = sanitize_key( $role_key );
					if ( isset( $editable_roles[ $role_key ] ) ) {
						$san['exempt_roles'][] = $role_key;
					}
				}
				$san['exempt_roles'] = array_values( array_unique( $san['exempt_roles'] ) );
			}

			// Bypass paths.
			$bypass_raw   = isset( $input['bypass_paths'] ) ? wp_unslash( $input['bypass_paths'] ) : '';
			$bypass_lines = preg_split( '/[\r\n]+/', $bypass_raw );
			$bypass_clean = array();
			if ( is_array( $bypass_lines ) ) {
				foreach ( $bypass_lines as $line ) {
					$line = trim( $line );
					if ( '' !== $line ) {
						// Allow wildcard *, basic path structure.
						$bypass_clean[] = sanitize_text_field( $line );
					}
				}
			}
			$san['bypass_paths'] = implode( "\n", $bypass_clean );

			// Remote API enabled.
			$san['use_remote_api'] = ! empty( $input['use_remote_api'] ) ? 1 : 0;

			// Cache TTL (seconds).
			$cache_ttl = isset( $input['cache_ttl'] ) ? absint( $input['cache_ttl'] ) : $defaults['cache_ttl'];
			if ( $cache_ttl < 300 ) {
				$cache_ttl = 300; // Minimum 5 minutes.
			} elseif ( $cache_ttl > DAY_IN_SECONDS ) {
				$cache_ttl = DAY_IN_SECONDS; // Max 1 day.
			}
			$san['cache_ttl'] = $cache_ttl;

			// Block status.
			$allowed_statuses = array( 403, 451, 302 );
			$block_status     = isset( $input['block_status'] ) ? absint( $input['block_status'] ) : $defaults['block_status'];
			if ( ! in_array( $block_status, $allowed_statuses, true ) ) {
				$block_status = $defaults['block_status'];
			}
			$san['block_status'] = $block_status;

			// Block redirect URL (only meaningful if 302).
			$block_redirect        = isset( $input['block_redirect'] ) ? trim( $input['block_redirect'] ) : '';
			$san['block_redirect'] = '' !== $block_redirect ? esc_url_raw( $block_redirect ) : '';

			// Block message (HTML allowed, sanitized).
			$block_message           = isset( $input['block_message'] ) ? wp_kses_post( wp_unslash( $input['block_message'] ) ) : $defaults['block_message'];
			$san['block_message']    = $block_message;

			return $san;
		}

		/**
		 * Add settings page.
		 */
		public function add_settings_page() {
			add_options_page(
				__( 'Country Gate', 'dp-country-gate' ),
				__( 'Country Gate', 'dp-country-gate' ),
				'manage_options',
				'dds_country_gate',
				array( $this, 'render_settings_page' )
			);
		}

		/**
		 * Render settings page.
		 */
		public function render_settings_page() {
			if ( ! current_user_can( 'manage_options' ) ) {
				return;
			}
			?>
			<div class="wrap">
				<h1><?php echo esc_html__( 'DP Country Gate', 'dp-country-gate' ); ?></h1>

				<form method="post" action="options.php">
					<?php
					settings_fields( 'dds_country_gate' );
					do_settings_sections( 'dds_country_gate' );
					submit_button();
					?>
				</form>
			</div>
			<?php
		}

		/**
		 * Field: Mode.
		 */
		public function field_mode() {
			$opts = $this->get_options();
			?>
			<select name="<?php echo esc_attr( self::OPT_KEY ); ?>[mode]">
				<option value="blocklist" <?php selected( $opts['mode'], 'blocklist' ); ?>>
					<?php esc_html_e( 'Block listed countries', 'dp-country-gate' ); ?>
				</option>
				<option value="allowlist" <?php selected( $opts['mode'], 'allowlist' ); ?>>
					<?php esc_html_e( 'Allow only listed countries', 'dp-country-gate' ); ?>
				</option>
			</select>
			<p class="description">
				<?php esc_html_e( 'In blocklist mode, listed countries are blocked. In allowlist mode, only listed countries are allowed and all others are blocked.', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Countries.
		 */
		public function field_countries() {
			$opts      = $this->get_options();
			$countries = $opts['countries'];
			?>
			<textarea
				name="<?php echo esc_attr( self::OPT_KEY ); ?>[countries]"
				rows="3"
				cols="50"
			><?php echo esc_textarea( $countries ); ?></textarea>
			<p class="description">
				<?php esc_html_e( 'Enter a comma-separated list of ISO 3166-1 alpha-2 country codes (e.g. US, CA, GB).', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Allow logged-in users.
		 */
		public function field_allow_logged_in() {
			$opts = $this->get_options();
			?>
			<label>
				<input
					type="checkbox"
					name="<?php echo esc_attr( self::OPT_KEY ); ?>[allow_logged_in]"
					value="1"
					<?php checked( $opts['allow_logged_in'], 1 ); ?>
				/>
				<?php esc_html_e( 'Always allow logged-in users (ignore country gate for them).', 'dp-country-gate' ); ?>
			</label>
			<?php
		}

		/**
		 * Field: Exempt roles.
		 */
		public function field_exempt_roles() {
			$opts           = $this->get_options();
			$selected_roles = isset( $opts['exempt_roles'] ) && is_array( $opts['exempt_roles'] ) ? $opts['exempt_roles'] : array();
			$editable_roles = function_exists( 'get_editable_roles' ) ? get_editable_roles() : array();

			if ( empty( $editable_roles ) ) {
				echo '<p class="description">' . esc_html__( 'No editable roles found.', 'dp-country-gate' ) . '</p>';
				return;
			}

			foreach ( $editable_roles as $role_key => $role ) :
				?>
				<label style="display:block;margin-bottom:2px;">
					<input
						type="checkbox"
						name="<?php echo esc_attr( self::OPT_KEY ); ?>[exempt_roles][]"
						value="<?php echo esc_attr( $role_key ); ?>"
						<?php checked( in_array( $role_key, $selected_roles, true ) ); ?>
					/>
					<?php echo esc_html( translate_user_role( $role['name'] ) ); ?>
				</label>
				<?php
			endforeach;
			?>
			<p class="description">
				<?php esc_html_e( 'Users with these roles will always be allowed, even if their country would normally be blocked.', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Bypass paths.
		 */
		public function field_bypass_paths() {
			$opts   = $this->get_options();
			$paths  = $opts['bypass_paths'];
			?>
			<textarea
				name="<?php echo esc_attr( self::OPT_KEY ); ?>[bypass_paths]"
				rows="4"
				cols="50"
			><?php echo esc_textarea( $paths ); ?></textarea>
			<p class="description">
				<?php esc_html_e( 'One path per line. Use * as a wildcard. For example: /wp-login.php, /wp-admin/*, /wp-json/*', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Use remote API.
		 */
		public function field_use_remote_api() {
			$opts = $this->get_options();
			?>
			<label>
				<input
					type="checkbox"
					name="<?php echo esc_attr( self::OPT_KEY ); ?>[use_remote_api]"
					value="1"
					<?php checked( $opts['use_remote_api'], 1 ); ?>
				/>
				<?php esc_html_e( 'Use ipapi.co as a fallback to detect visitor country when Cloudflare or GEOIP headers are not available.', 'dp-country-gate' ); ?>
			</label>
			<p class="description">
				<?php esc_html_e( 'If enabled, visitor IP addresses are sent to ipapi.co. Only the returned country code is stored in a transient cache.', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Cache TTL.
		 */
		public function field_cache_ttl() {
			$opts      = $this->get_options();
			$cache_ttl = isset( $opts['cache_ttl'] ) ? absint( $opts['cache_ttl'] ) : 3600;
			?>
			<input
				type="number"
				min="300"
				max="<?php echo esc_attr( DAY_IN_SECONDS ); ?>"
				name="<?php echo esc_attr( self::OPT_KEY ); ?>[cache_ttl]"
				value="<?php echo esc_attr( $cache_ttl ); ?>"
			/>
			<p class="description">
				<?php esc_html_e( 'Cache lifetime in seconds for remote IP lookups (minimum 300, maximum 86400).', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Block status.
		 */
		public function field_block_status() {
			$opts   = $this->get_options();
			$status = isset( $opts['block_status'] ) ? absint( $opts['block_status'] ) : 403;
			?>
			<select name="<?php echo esc_attr( self::OPT_KEY ); ?>[block_status]">
				<option value="403" <?php selected( $status, 403 ); ?>>
					<?php esc_html_e( '403 Forbidden', 'dp-country-gate' ); ?>
				</option>
				<option value="451" <?php selected( $status, 451 ); ?>>
					<?php esc_html_e( '451 Unavailable For Legal Reasons', 'dp-country-gate' ); ?>
				</option>
				<option value="302" <?php selected( $status, 302 ); ?>>
					<?php esc_html_e( '302 Redirect', 'dp-country-gate' ); ?>
				</option>
			</select>
			<p class="description">
				<?php esc_html_e( 'Choose the HTTP status code used when blocking visitors. If 302 is selected, you can specify a redirect URL below.', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Block redirect.
		 */
		public function field_block_redirect() {
			$opts     = $this->get_options();
			$redirect = isset( $opts['block_redirect'] ) ? $opts['block_redirect'] : '';
			?>
			<input
				type="url"
				name="<?php echo esc_attr( self::OPT_KEY ); ?>[block_redirect]"
				value="<?php echo esc_attr( $redirect ); ?>"
				class="regular-text"
			/>
			<p class="description">
				<?php esc_html_e( 'If block response type is 302, visitors will be redirected to this URL. Leave empty to use a normal error page.', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Field: Block message.
		 */
		public function field_block_message() {
			$opts    = $this->get_options();
			$message = isset( $opts['block_message'] ) ? $opts['block_message'] : '';
			?>
			<textarea
				name="<?php echo esc_attr( self::OPT_KEY ); ?>[block_message]"
				rows="6"
				cols="60"
			><?php echo esc_textarea( $message ); ?></textarea>
			<p class="description">
				<?php esc_html_e( 'HTML is allowed. This message will be shown when a visitor is blocked (for 403/451 responses).', 'dp-country-gate' ); ?>
			</p>
			<?php
		}

		/**
		 * Main check: maybe block the visitor on front-end.
		 */
		public function maybe_block_visitor() {
			// Only affect front-end requests.
			if ( is_admin() && ! wp_doing_ajax() ) {
				return;
			}

			$opts = $this->get_options();

			// Bypass paths.
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( $_SERVER['REQUEST_URI'] ) : '';
			$path        = parse_url( $request_uri, PHP_URL_PATH );
			if ( $this->is_bypass_path( $path, $opts ) ) {
				return;
			}

			// Logged-in users allowed?
			if ( is_user_logged_in() ) {
				if ( ! empty( $opts['allow_logged_in'] ) ) {
					return;
				}

				// Or role-based exemption.
				if ( ! empty( $opts['exempt_roles'] ) && is_array( $opts['exempt_roles'] ) ) {
					$user = wp_get_current_user();
					if ( ! empty( $user->roles ) ) {
						foreach ( $user->roles as $role ) {
							if ( in_array( $role, $opts['exempt_roles'], true ) ) {
								return;
							}
						}
					}
				}
			}

			// If no countries configured, do nothing.
			if ( empty( $opts['countries'] ) ) {
				return;
			}

			// Detect country.
			$country = $this->detect_country( $opts );

			/**
			 * Filter the detected visitor country code.
			 *
			 * @since 1.6.1
			 *
			 * @param string $country Two-letter country code or empty string.
			 * @param array  $opts    Plugin options.
			 */
			$country = apply_filters( 'dp_country_gate_detected_country', $country, $opts );

			if ( empty( $country ) ) {
				// If we cannot detect a country, fail open (allow).
				return;
			}

			$defined_countries = array_map(
				'trim',
				explode( ',', strtoupper( $opts['countries'] ) )
			);
			$defined_countries = array_filter(
				$defined_countries,
				function ( $code ) {
					return ( 2 === strlen( $code ) );
				}
			);

			if ( empty( $defined_countries ) ) {
				return;
			}

			$in_list = in_array( strtoupper( $country ), $defined_countries, true );

			$should_block = false;

			if ( 'blocklist' === $opts['mode'] && $in_list ) {
				$should_block = true;
			} elseif ( 'allowlist' === $opts['mode'] && ! $in_list ) {
				$should_block = true;
			}

			/**
			 * Filter whether the current visitor should be blocked.
			 *
			 * @since 1.6.1
			 *
			 * @param bool   $should_block      Whether to block this visitor.
			 * @param string $country           Detected country code.
			 * @param array  $defined_countries Configured list of country codes.
			 * @param array  $opts              Plugin options.
			 */
			$should_block = apply_filters(
				'dp_country_gate_should_block',
				$should_block,
				$country,
				$defined_countries,
				$opts
			);

			if ( $should_block ) {
				$this->execute_block( $opts );
			}
		}

		/**
		 * Check if a path should bypass the gate.
		 *
		 * @param string $path Request path.
		 * @param array  $opts Options.
		 *
		 * @return bool
		 */
		protected function is_bypass_path( $path, $opts ) {
			if ( empty( $path ) ) {
				return false;
			}

			$paths_config = isset( $opts['bypass_paths'] ) ? $opts['bypass_paths'] : '';
			if ( '' === trim( $paths_config ) ) {
				return false;
			}

			$lines = preg_split( '/[\r\n]+/', $paths_config );
			if ( ! is_array( $lines ) ) {
				return false;
			}

			foreach ( $lines as $line ) {
				$pattern = trim( $line );
				if ( '' === $pattern ) {
					continue;
				}

				// Convert wildcard pattern to a regular expression.
				$regex = preg_quote( $pattern, '/' );
				$regex = str_replace( '\*', '.*', $regex );

				if ( preg_match( '/^' . $regex . '$/i', $path ) ) {
					return true;
				}
			}

			return false;
		}

		/**
		 * Detect visitor country using headers, server vars, or remote API.
		 *
		 * @param array $opts Options.
		 *
		 * @return string Two-letter country code or empty string.
		 */
		protected function detect_country( $opts ) {
			// 1. Cloudflare header.
			if ( ! empty( $_SERVER['HTTP_CF_IPCOUNTRY'] ) ) {
				$cf = strtoupper( sanitize_text_field( wp_unslash( $_SERVER['HTTP_CF_IPCOUNTRY'] ) ) );
				if ( 2 === strlen( $cf ) && 'XX' !== $cf ) {
					return $cf;
				}
			}

			// 2. Common GEOIP server variables.
			$geo_vars = array(
				'GEOIP_COUNTRY_CODE',
				'HTTP_X_GEOIP2_COUNTRY',
				'HTTP_CF_IPCOUNTRY', // already checked, but kept for completeness.
			);

			foreach ( $geo_vars as $var ) {
				if ( ! empty( $_SERVER[ $var ] ) ) {
					$code = strtoupper( sanitize_text_field( wp_unslash( $_SERVER[ $var ] ) ) );
					if ( 2 === strlen( $code ) && ctype_alpha( $code ) ) {
						return $code;
					}
				}
			}

			// 3. Remote API fallback.
			if ( ! empty( $opts['use_remote_api'] ) ) {
				return $this->detect_country_remote( $opts );
			}

			return '';
		}

		/**
		 * Detect country via remote API (ipapi.co) with transient caching.
		 *
		 * @param array $opts Options.
		 *
		 * @return string
		 */
		protected function detect_country_remote( $opts ) {
			$ip = $this->get_client_ip();
			if ( empty( $ip ) ) {
				return '';
			}

			$transient_key = self::TRANSIENT_PREFIX . md5( $ip );
			$cached        = get_transient( $transient_key );
			if ( is_string( $cached ) && 2 === strlen( $cached ) ) {
				return strtoupper( $cached );
			}

			// Make a remote call using WordPress HTTP API.
			$url  = 'https://ipapi.co/' . rawurlencode( $ip ) . '/country/';
			$args = array(
				'timeout' => 3,
			);

			$response = wp_remote_get( $url, $args );
			if ( is_wp_error( $response ) ) {
				return '';
			}

			$code = wp_remote_retrieve_body( $response );
			$code = strtoupper( trim( (string) $code ) );

			if ( 2 !== strlen( $code ) || ! ctype_alpha( $code ) ) {
				return '';
			}

			$ttl = isset( $opts['cache_ttl'] ) ? absint( $opts['cache_ttl'] ) : 3600;
			if ( $ttl < 300 ) {
				$ttl = 300;
			} elseif ( $ttl > DAY_IN_SECONDS ) {
				$ttl = DAY_IN_SECONDS;
			}

			set_transient( $transient_key, $code, $ttl );

			return $code;
		}

		/**
		 * Get client IP address (best-effort).
		 *
		 * @return string
		 */
		protected function get_client_ip() {
			$ip_keys = array(
				'HTTP_CF_CONNECTING_IP',
				'HTTP_X_REAL_IP',
				'HTTP_X_FORWARDED_FOR',
				'REMOTE_ADDR',
			);

			foreach ( $ip_keys as $key ) {
				if ( empty( $_SERVER[ $key ] ) ) {
					continue;
				}

				$value = wp_unslash( $_SERVER[ $key ] );

				if ( 'HTTP_X_FORWARDED_FOR' === $key ) {
					// Could be a comma-separated list.
					$parts = explode( ',', (string) $value );
					$value = trim( reset( $parts ) );
				}

				$value = trim( (string) $value );

				if ( filter_var( $value, FILTER_VALIDATE_IP ) ) {
					return $value;
				}
			}

			return '';
		}

		/**
		 * Execute the blocking action for the visitor.
		 *
		 * @param array $opts Options.
		 */
		protected function execute_block( $opts ) {
			$status = isset( $opts['block_status'] ) ? absint( $opts['block_status'] ) : 403;
			if ( ! in_array( $status, array( 403, 451, 302 ), true ) ) {
				$status = 403;
			}

			/**
			 * Filter the response status code used when blocking a visitor.
			 *
			 * @since 1.6.1
			 *
			 * @param int   $status HTTP status code (403, 451, or 302).
			 * @param array $opts   Plugin options.
			 */
			$status = (int) apply_filters( 'dp_country_gate_block_status', $status, $opts );

			// Redirect if 302 and we have a valid URL.
			if ( 302 === $status && ! empty( $opts['block_redirect'] ) ) {
				wp_safe_redirect( $opts['block_redirect'], 302 );
				exit;
			}

			// Otherwise, show an error page via wp_die.
			$message = isset( $opts['block_message'] ) ? wp_kses_post( $opts['block_message'] ) : '';
			if ( '' === $message ) {
				$message = esc_html__( 'Access restricted from your country.', 'dp-country-gate' );
			}

			$title = esc_html__( 'Access Restricted', 'dp-country-gate' );

			wp_die(
				$message,
				$title,
				array(
					'response' => $status,
				)
			);
		}

		/**
		 * Uninstall callback. Remove plugin options.
		 */
		public static function uninstall() {
			delete_option( self::OPT_KEY );
			// Transients will naturally expire; no need for additional cleanup.
		}
	}
}

/**
 * Register uninstall hook.
 */
if ( class_exists( 'DDS_Country_Gate' ) ) {
	register_uninstall_hook(
		__FILE__,
		array( 'DDS_Country_Gate', 'uninstall' )
	);

	// Bootstrap plugin.
	new DDS_Country_Gate();
}
